/*
 * File: BankFailure.java
 * =============================================================
 * A class representing a failure of a single bank.
 */

import acm.util.*;
import java.util.*;
import java.text.*;

public class BankFailure {
	/* The bank that failed. */
	private String failedBank;
	
	/* What city it is in. */
	private String city;
	
	/* What state that bank is in. */
	private String state;
	
	/* Who acquired that bank. */
	private String acquirer;
	
	/* The date that the bank failed. */
	private Date failDate;
	
	/**
	 * Given a line from the Bank Failures File, parses that line into a
	 * BankFailure object.
	 * 
	 * @param line The line from the file.
	 */
	public BankFailure(String line) {
		/* Split the string into pieces at tab boundaries. */
		String[] pieces = line.split("\t");
		
		/* Pick up the first few pieces. */
		failedBank = pieces[0];
		city = pieces[1];
		state = pieces[2];
		acquirer = pieces[4];
		
		/* Parse the failure date from the last piece. */
		try {
			DateFormat format = new SimpleDateFormat("dd-MMM-yy");
			failDate = format.parse(pieces[5]);
		} catch (ParseException p) {
			throw new ErrorException(p);
		}
	}
	
	/**
	 * Returns the date at which the bank failed.
	 * 
	 * @return The date at which the bank failed.
	 */
	public Date getFailDate() {
		return failDate;
	}
	
	/**
	 * Returns the city of the failed bank.
	 * 
	 * @return The city of the failed bank.
	 */
	public String getCity() {
		return city;
	}
	
	/**
	 * Returns the state of the failed bank.
	 * 
	 * @return The state of the failed bank.
	 */
	public String getState() {
		return state;
	}
	
	/**
	 * Returns the name of the bank that failed.
	 * 
	 * @return The name of the bank that failed.
	 */
	public String getBankName() {
		return failedBank;
	}
	
	/**
	 * Returns the name of the bank that acquired the bank that failed.
	 * 
	 * @return The name of the bank that acquired the bank that failed.
	 */
	public String getAcquirer() {
		return acquirer;
	}
}
